#==============================================================================
# Contains miscellaneous scaling-related utility procedures.
#
# Structure of the module:
#   - Namespace initialization
#   - Public utility procedures
#   - Private helper procedures
#
# Copyright (c) 2020-2024  Csaba Nemethi (E-mail: csaba.nemethi@t-online.de)
#==============================================================================

if {[catch {package require Tk 8.4-}]} {
    package require Tk 8.4
}
if {[catch {package require scaleutil 1.10-}]} {
    package require scaleutil 1.10
}

#
# Namespace initialization
# ========================
#

namespace eval scaleutilmisc {
    #
    # Public variables:
    #
    variable version	1.8
    variable library	[file dirname [file normalize [info script]]]

    #
    # Public procedures:
    #
    namespace export	scaleBWidgetSpinBox scaleBWidgetComboBox \
			scaleIncrDateentry scaleIncrTimeentry \
			scaleIncrCombobox scaleOakleyComboboxArrow

    proc getScalingPct {} {
	set pct [expr {[tk scaling] * 75}]
	for {set intPct 100} {1} {incr intPct 25} {
	    if {$pct < $intPct + 12.5} {
		return $intPct
	    }
	}
    }

    variable onX11 [expr {[tk windowingsystem] eq "x11"}]

    variable svgSupported \
	[expr {$::tk_version >= 8.7 || [catch {package require tksvg}] == 0}]
}

package provide scaleutilmisc $scaleutilmisc::version

#
# Public utility procedures
# =========================
#

#------------------------------------------------------------------------------
# scaleutilmisc::scaleBWidgetSpinBox
#
# Scales a BWidget SpinBox widget of the given path name.
#------------------------------------------------------------------------------
proc scaleutilmisc::scaleBWidgetSpinBox w {
    #
    # Scale the width of the two arrows, which is set to 11
    #
    set arrWidth [::scaleutil::scale 11 [getScalingPct]]
    $w.arrup configure -width $arrWidth
    $w.arrdn configure -width $arrWidth
}

#------------------------------------------------------------------------------
# scaleutilmisc::scaleBWidgetComboBox
#
# Scales a BWidget ComboBox widget of the given path name.
#------------------------------------------------------------------------------
proc scaleutilmisc::scaleBWidgetComboBox w {
    #
    # Scale the width of the arrow, which is set to 11 or 15
    #
    variable onX11
    set defaultWidth [expr {$onX11 ? 11 : 15}]
    set width [::scaleutil::scale $defaultWidth [getScalingPct]]
    $w.a configure -width $width

    #
    # Scale the width of the two scrollbars, which is set to 11 or 15
    #
    ComboBox::_create_popup $w
    if {![Widget::theme]} {
	##nagelfar ignore
	bind $w.shell <Map> [format {
	    if {[winfo class %%W] eq "Toplevel"} {
		%%W.sw.vscroll configure -width %d
		%%W.sw.hscroll configure -width %d
	    }
	} $width $width]
    }
}

#------------------------------------------------------------------------------
# scaleutilmisc::scaleIncrDateentry
#
# Scales an [incr Widgets] dateentry of the given path name.
#------------------------------------------------------------------------------
proc scaleutilmisc::scaleIncrDateentry {w pct} {
    #
    # Scale the values of a few options
    #
    set btnFg [$w cget -buttonforeground]
    $w configure -icon [calendarImg $pct] -selectthickness 2p \
	-backwardimage [backwardImg $pct $btnFg] \
	-forwardimage  [forwardImg  $pct $btnFg]
    variable onX11
    if {$onX11 && $::tk_version >= 8.5} {
	set captionFontSize [font actual TkCaptionFont -size]
	set defaultFontSize [font actual TkDefaultFont -size]
	$w configure -titlefont [list Helvetica $captionFontSize bold] \
	    -dayfont [list Helvetica $defaultFontSize] \
	    -datefont [list Helvetica $defaultFontSize]\
	    -currentdatefont [list Helvetica $defaultFontSize bold]
    } else {
	$w configure -titlefont {Helvetica 11 bold} -dayfont {Helvetica 9} \
	    -datefont {Helvetica 9} -currentdatefont {Helvetica 9 bold}
    }
    set default [lindex [$w configure -height] 3]
    $w configure -height [::scaleutil::scale $default $pct]
    set default [lindex [$w configure -width] 3]
    $w configure -width [::scaleutil::scale $default $pct]
}

#------------------------------------------------------------------------------
# scaleutilmisc::scaleIncrTimeentry
#
# Scales an [incr Widgets] timeentry of the given path name.
#------------------------------------------------------------------------------
proc scaleutilmisc::scaleIncrTimeentry {w pct} {
    #
    # Scale the values of a few options
    #
    $w configure -icon [watchImg $pct]
    set default [lindex [$w configure -watchheight] 3]
    $w configure -watchheight [::scaleutil::scale $default $pct]
    set default [lindex [$w configure -watchwidth] 3]
    $w configure -watchwidth [::scaleutil::scale $default $pct]
}

#------------------------------------------------------------------------------
# scaleutilmisc::scaleIncrCombobox
#
# Scales an [incr Widgets] combobox of the given path name.
#------------------------------------------------------------------------------
proc scaleutilmisc::scaleIncrCombobox {w pct} {
    #
    # Scale the two arrows, as well as the value of the -listheight
    # option and that of the -sbwidth option of the list component
    #
    scaleIncrComboboxArrows $pct
    set default [lindex [$w configure -listheight] 3]
    $w configure -listheight [::scaleutil::scale $default $pct]
    set listComp [$w component list]
    set default [lindex [$listComp configure -sbwidth] 3]
    $listComp configure -sbwidth [::scaleutil::scale $default $pct]
}

#------------------------------------------------------------------------------
# scaleutilmisc::scaleOakleyComboboxArrow
#
# Scales the default arrow of the Oakley combobox widget.
#------------------------------------------------------------------------------
proc scaleutilmisc::scaleOakleyComboboxArrow pct {
    switch $pct {
	100 {
	    set data "
#define down_width 9
#define down_height 5
static unsigned char down_bits[] = {
   0xff, 0x01, 0xfe, 0x00, 0x7c, 0x00, 0x38, 0x00, 0x10, 0x00};"
	}
	125 {
	    set data "
#define down_width 11
#define down_height 6
static unsigned char down_bits[] = {
   0xff, 0x07, 0xfe, 0x03, 0xfc, 0x01, 0xf8, 0x00, 0x70, 0x00, 0x20, 0x00};"
	}
	150 {
	    set data "
#define down_width 13
#define down_height 7
static unsigned char down_bits[] = {
   0xff, 0x1f, 0xfe, 0x0f, 0xfc, 0x07, 0xf8, 0x03, 0xf0, 0x01, 0xe0, 0x00,
   0x40, 0x00};"
	}
	175 {
	    set data "
#define down_width 15
#define down_height 8
static unsigned char down_bits[] = {
   0xff, 0x7f, 0xfe, 0x3f, 0xfc, 0x1f, 0xf8, 0x0f, 0xf0, 0x07, 0xe0, 0x03,
   0xc0, 0x01, 0x80, 0x00};"
	}
	200 {
	    set data "
#define down_width 17
#define down_height 9
static unsigned char down_bits[] = {
   0xff, 0xff, 0x01, 0xfe, 0xff, 0x00, 0xfc, 0x7f, 0x00, 0xf8, 0x3f, 0x00,
   0xf0, 0x1f, 0x00, 0xe0, 0x0f, 0x00, 0xc0, 0x07, 0x00, 0x80, 0x03, 0x00,
   0x00, 0x01, 0x00};"
	}
    }

    ::combobox::bimage configure -data $data
}

#
# Private helper procedures
# =========================
#

#------------------------------------------------------------------------------
# scaleutilmisc::calendarImg
#------------------------------------------------------------------------------
proc scaleutilmisc::calendarImg pct {
    variable calendarImg
    if {[info exists calendarImg]} {
	return $calendarImg
    }

    set calendarImg [image create photo scaleutilmisc_calendarImg]

    variable svgSupported
    if {$svgSupported} {
	set svgFmt [list svg -scale [expr {[getScalingPct] / 100.0}]]
	scaleutilmisc_calendarImg put {
<svg width="16" height="16" version="1.1" xmlns="http://www.w3.org/2000/svg">
 <rect width="16" height="16" rx="1" fill="#8b0000"/>
 <rect x="1" y="5" width="14" height="10" fill="#fff"/>
 <rect x="6" y="6" width="2" height="2"/>
 <rect x="9" y="6" width="2" height="2"/>
 <rect x="12" y="6" width="2" height="2"/>
 <rect x="3" y="6" width="2" height="2"/>
 <rect x="3" y="9" width="2" height="2"/>
 <rect x="6" y="9" width="2" height="2"/>
 <rect x="9" y="9" width="2" height="2"/>
 <rect x="12" y="9" width="2" height="2"/>
 <rect x="3" y="12" width="2" height="2"/>
 <rect x="6" y="12" width="2" height="2"/>
 <rect x="9" y="12" width="2" height="2"/>
 <rect x="12" y="12" width="2" height="2"/>
</svg>
	} -format $svgFmt

    } else {
	switch $pct {
	    100 {
		set data "
R0lGODlhEAAQAKECAAAAAIsAAP///////yH5BAEKAAMALAAAAAAQABAAAAIojI+py+0Pj5i0VjMB
EJrvjnliBwbWSYXkxwpqu7omer5ja8clXUdHAQA7"
	    }
	    125 {
		set data "
R0lGODlhFAAUAOMMAAAAAIoAAIsAAD8/P29vb39/f82QkJ+fn7+/v+LAwM/Pz9/f3///////////
/////yH5BAEKAA8ALAAAAAAUABQAAAR5UMhJKw026827/2BnJGRpnokhJcxSFAvyMkdxMC+SrAwC
ALIfYwAYMH6FnYDlAxaERCNSyWJYr9gslaEYDBQHL4MwIAwHh20zCBgWj4Akr/sNG8lmb5qX7WPV
P2xuUnGATlBvUzwuMDIFNDY4BToSIyiXJCoYIRIBEQA7"
	    }
	    150 {
		set data "
R0lGODlhGAAYAOMHAAAAAIoAAIsAAKhAQH9/f8WAgL+/v///////////////////////////////
/////yH5BAEKAAgALAAAAAAYABgAAAR4UMhJq5Xh6s27/2Aojt9QnGiqqoNUHHAsz3PhwgQAEIcO
HLldT3ewCV5AHc+XFDKNyOCSKB3+oLRsFnswEAgG4Lf7DX953Cpg3Zxeb21rfP0+wrxgMQ9vHnO1
gEVwamyEdVFrbnNsaYlyhoJ2gYBGJiuXKy0ZJBQRADs="
	    }
	    175 {
		set data "
R0lGODlhHAAcAOMLAAAAAIsAAJIQED8/P6hAQH9/f5+fn7+/v8/Pz9/f3+/v7///////////////
/////yH5BAEKAA8ALAAAAAAcABwAAAS48IVJq71W4s27/2AojmRpnphArGzrvqwwEUtt33huE7Nt
FAXD4gAsLBLFxKJ4WPACtNoAABgsqYAhtok1PqOLafVK1VK5VG9PSrUWsGYAGqCG6u74rw1xOCCO
fU0KgQpDfUp6bGNvZQdbC11Oa2FtZFmOZ5BpknaKbnCYcpp0nGB8foB9C4N9hYGIk3iyNYmUi6CP
kbVin425m7uVjJe/pLU/QUNFR0lLQE21s7JPKjDW1jIaKBQPEQA7"
	    }
	    200 {
		set data "
R0lGODlhIAAgAKECAAAAAIsAAP///////yH5BAEKAAMALAAAAAAgACAAAAJvXI6py2cNo5y02ouz
3rz7jwniSJbmmZzqSqYjAANiLAv0HIsuDvP1bcsJdsFe8ScEEoHMpJDIio6g0ujSaWwar1kscosg
ab9kXfjl9amH52qV6lZxy8f1fD02H8Tpuv/ul8e2FycFV1gCAvGg6FAAADs="
	    }
	}
	scaleutilmisc_calendarImg put $data -format gif
    }

    return $calendarImg
}

#------------------------------------------------------------------------------
# scaleutilmisc::backwardImg
#------------------------------------------------------------------------------
proc scaleutilmisc::backwardImg {pct fg} {
    variable backwardImg
    if {![info exists backwardImg]} {
	set backwardImg [image create bitmap scaleutilmisc_backwardImg]

	switch $pct {
	    100 {
		set data "
#define backward_width 16
#define backward_height 16
static unsigned char backward_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x30, 0xe0, 0x38, 0xf0, 0x3c,
   0xf8, 0x3e, 0xfc, 0x3f, 0xfc, 0x3f, 0xf8, 0x3e, 0xf0, 0x3c, 0xe0, 0x38,
   0xc0, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	    }
	    125 {
		set data "
#define backward_width 20
#define backward_height 20
static unsigned char backward_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x03,
   0x80, 0x83, 0x03, 0xc0, 0xc3, 0x03, 0xe0, 0xe3, 0x03, 0xf0, 0xf3, 0x03,
   0xf8, 0xfb, 0x03, 0xfc, 0xff, 0x03, 0xfc, 0xff, 0x03, 0xf8, 0xfb, 0x03,
   0xf0, 0xf3, 0x03, 0xe0, 0xe3, 0x03, 0xc0, 0xc3, 0x03, 0x80, 0x83, 0x03,
   0x00, 0x03, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	    }
	    150 {
		set data "
#define backward_width 24
#define backward_height 24
static unsigned char backward_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x0c, 0x18, 0x00, 0x0e, 0x1c, 0x00, 0x0f, 0x1e, 0x80, 0x0f, 0x1f,
   0xc0, 0x8f, 0x1f, 0xe0, 0xcf, 0x1f, 0xf0, 0xef, 0x1f, 0xf8, 0xff, 0x1f,
   0xf8, 0xff, 0x1f, 0xf0, 0xef, 0x1f, 0xe0, 0xcf, 0x1f, 0xc0, 0x8f, 0x1f,
   0x80, 0x0f, 0x1f, 0x00, 0x0f, 0x1e, 0x00, 0x0e, 0x1c, 0x00, 0x0c, 0x18,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	    }
	    175 {
		set data "
#define backward_width 28
#define backward_height 28
static unsigned char backward_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x80, 0x01, 0x00, 0x38, 0xc0, 0x01,
   0x00, 0x3c, 0xe0, 0x01, 0x00, 0x3e, 0xf0, 0x01, 0x00, 0x3f, 0xf8, 0x01,
   0x80, 0x3f, 0xfc, 0x01, 0xc0, 0x3f, 0xfe, 0x01, 0xe0, 0x3f, 0xff, 0x01,
   0xf0, 0xbf, 0xff, 0x01, 0xf8, 0xff, 0xff, 0x01, 0xf8, 0xff, 0xff, 0x01,
   0xf0, 0xbf, 0xff, 0x01, 0xe0, 0x3f, 0xff, 0x01, 0xc0, 0x3f, 0xfe, 0x01,
   0x80, 0x3f, 0xfc, 0x01, 0x00, 0x3f, 0xf8, 0x01, 0x00, 0x3e, 0xf0, 0x01,
   0x00, 0x3c, 0xe0, 0x01, 0x00, 0x38, 0xc0, 0x01, 0x00, 0x30, 0x80, 0x01,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00};"
	    }
	    200 {
		set data "
#define backward_width 32
#define backward_height 32
static unsigned char backward_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x00, 0x0e,
   0x00, 0xf0, 0x00, 0x0f, 0x00, 0xf8, 0x80, 0x0f, 0x00, 0xfc, 0xc0, 0x0f,
   0x00, 0xfe, 0xe0, 0x0f, 0x00, 0xff, 0xf0, 0x0f, 0x80, 0xff, 0xf8, 0x0f,
   0xc0, 0xff, 0xfc, 0x0f, 0xe0, 0xff, 0xfe, 0x0f, 0xf0, 0xff, 0xff, 0x0f,
   0xf0, 0xff, 0xff, 0x0f, 0xf0, 0xff, 0xff, 0x0f, 0xf0, 0xff, 0xff, 0x0f,
   0xe0, 0xff, 0xfe, 0x0f, 0xc0, 0xff, 0xfc, 0x0f, 0x80, 0xff, 0xf8, 0x0f,
   0x00, 0xff, 0xf0, 0x0f, 0x00, 0xfe, 0xe0, 0x0f, 0x00, 0xfc, 0xc0, 0x0f,
   0x00, 0xf8, 0x80, 0x0f, 0x00, 0xf0, 0x00, 0x0f, 0x00, 0xe0, 0x00, 0x0e,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	    }
	}
	scaleutilmisc_backwardImg configure -data $data -foreground $fg
    }

    return $backwardImg
}

#------------------------------------------------------------------------------
# scaleutilmisc::forwardImg
#------------------------------------------------------------------------------
proc scaleutilmisc::forwardImg {pct fg} {
    variable forwardImg
    if {![info exists forwardImg]} {
	set forwardImg [image create bitmap scaleutilmisc_forwardImg]

	switch $pct {
	    100 {
		set data "
#define forward_width 16
#define forward_height 16
static unsigned char forward_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x03, 0x1c, 0x07, 0x3c, 0x0f,
   0x7c, 0x1f, 0xfc, 0x3f, 0xfc, 0x3f, 0x7c, 0x1f, 0x3c, 0x0f, 0x1c, 0x07,
   0x0c, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	    }
	    125 {
		set data "
#define forward_width 20
#define forward_height 20
static unsigned char forward_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x0c, 0x00,
   0x1c, 0x1c, 0x00, 0x3c, 0x3c, 0x00, 0x7c, 0x7c, 0x00, 0xfc, 0xfc, 0x00,
   0xfc, 0xfd, 0x01, 0xfc, 0xff, 0x03, 0xfc, 0xff, 0x03, 0xfc, 0xfd, 0x01,
   0xfc, 0xfc, 0x00, 0x7c, 0x7c, 0x00, 0x3c, 0x3c, 0x00, 0x1c, 0x1c, 0x00,
   0x0c, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	    }
	    150 {
		set data "
#define forward_width 24
#define forward_height 24
static unsigned char forward_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x18, 0x30, 0x00, 0x38, 0x70, 0x00, 0x78, 0xf0, 0x00, 0xf8, 0xf0, 0x01,
   0xf8, 0xf1, 0x03, 0xf8, 0xf3, 0x07, 0xf8, 0xf7, 0x0f, 0xf8, 0xff, 0x1f,
   0xf8, 0xff, 0x1f, 0xf8, 0xf7, 0x0f, 0xf8, 0xf3, 0x07, 0xf8, 0xf1, 0x03,
   0xf8, 0xf0, 0x01, 0x78, 0xf0, 0x00, 0x38, 0x70, 0x00, 0x18, 0x30, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	    }
	    175 {
		set data "
#define forward_width 28
#define forward_height 28
static unsigned char forward_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x18, 0xc0, 0x00, 0x00, 0x38, 0xc0, 0x01, 0x00,
   0x78, 0xc0, 0x03, 0x00, 0xf8, 0xc0, 0x07, 0x00, 0xf8, 0xc1, 0x0f, 0x00,
   0xf8, 0xc3, 0x1f, 0x00, 0xf8, 0xc7, 0x3f, 0x00, 0xf8, 0xcf, 0x7f, 0x00,
   0xf8, 0xdf, 0xff, 0x00, 0xf8, 0xff, 0xff, 0x01, 0xf8, 0xff, 0xff, 0x01,
   0xf8, 0xdf, 0xff, 0x00, 0xf8, 0xcf, 0x7f, 0x00, 0xf8, 0xc7, 0x3f, 0x00,
   0xf8, 0xc3, 0x1f, 0x00, 0xf8, 0xc1, 0x0f, 0x00, 0xf8, 0xc0, 0x07, 0x00,
   0x78, 0xc0, 0x03, 0x00, 0x38, 0xc0, 0x01, 0x00, 0x18, 0xc0, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00};"
	    }
	    200 {
		set data "
#define forward_width 32
#define forward_height 32
static unsigned char forward_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x00, 0x07, 0x00,
   0xf0, 0x00, 0x0f, 0x00, 0xf0, 0x01, 0x1f, 0x00, 0xf0, 0x03, 0x3f, 0x00,
   0xf0, 0x07, 0x7f, 0x00, 0xf0, 0x0f, 0xff, 0x00, 0xf0, 0x1f, 0xff, 0x01,
   0xf0, 0x3f, 0xff, 0x03, 0xf0, 0x7f, 0xff, 0x07, 0xf0, 0xff, 0xff, 0x0f,
   0xf0, 0xff, 0xff, 0x0f, 0xf0, 0xff, 0xff, 0x0f, 0xf0, 0xff, 0xff, 0x0f,
   0xf0, 0x7f, 0xff, 0x07, 0xf0, 0x3f, 0xff, 0x03, 0xf0, 0x1f, 0xff, 0x01,
   0xf0, 0x0f, 0xff, 0x00, 0xf0, 0x07, 0x7f, 0x00, 0xf0, 0x03, 0x3f, 0x00,
   0xf0, 0x01, 0x1f, 0x00, 0xf0, 0x00, 0x0f, 0x00, 0x70, 0x00, 0x07, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	    }
	}
	scaleutilmisc_forwardImg configure -data $data -foreground $fg
    }

    return $forwardImg
}

#------------------------------------------------------------------------------
# scaleutilmisc::watchImg
#------------------------------------------------------------------------------
proc scaleutilmisc::watchImg pct {
    variable watchImg
    if {[info exists watchImg]} {
	return $watchImg
    }

    set watchImg [image create photo scaleutilmisc_watchImg]

    variable svgSupported
    if {$svgSupported} {
	set svgFmt [list svg -scale [expr {[getScalingPct] / 100.0}]]
	scaleutilmisc_watchImg put {
<svg width="16" height="16" version="1.1" xmlns="http://www.w3.org/2000/svg">
 <circle cx="8" cy="8" r="7.5" fill="#fff" stroke="#8b0000"/>
 <path d="m8 3v5h4" fill="none" stroke="#000" stroke-linecap="round" stroke-linejoin="round"/>
</svg>
	} -format $svgFmt

    } else {
	switch $pct {
	    100 {
		set data "
R0lGODlhEAAQAKUlAI0FBY0GBo4GBpAMDJEMDJENDZEODpIODpEPD5IPD5IQEJMQEERERK1KSq1L
S65MTK5NTXd3d39/f4iIiMR+fsWAgMWBgcaCgsaDg9DQ0OjNzenOzunPz+rQ0Pjw8Pnx8fny8vrz
8/r09P79/f/+/v//////////////////////////////////////////////////////////////
/////////////////////////////////////////////yH5BAEKAD8ALAAAAAAQABAAAAaGwJ9Q
OFAAAIrBcPk7JC4ckWhzSSCYBwioVMpkuJ+HYZglcUsSyXnkuP4U23P6XPokfgMMHa2mVwYLHXtz
dBxGIYMME4tfIQIAcWcZEZQRXx8BCoJ7nBsLAxacnBRKCR+iZx5jPwgPI6gjDXdCBg6nex4Nq0MF
CBYaISEaFQizTD8EC0dJTEEAOw=="
	    }
	    125 {
		set data "
R0lGODlhFAAUAKUtAIsCAowCAowDA4wEBI0EBIwFBY0FBY0GBo4HB44ICCYmJo8ICI8JCY8KCpAK
CpALC5EMDGBgYGZmZrFSUrFTU7FUVLJVVbJWVrNXV8Bzc8B0dMF2dsJ3d8J5eYyMjM+WltCXl9CY
mLS0tNyxsdyystyzs920tOC6uuG8vOG+vvfu7vjv7/jw8P//////////////////////////////
/////////////////////////////////////////////yH5BAEKAD8ALAAAAAAUABQAAAatwJ9w
+Cs4CARHgcgcJhiXUIkUsjAQTSFB0GG1vuAVRwBgEiApcMvjUaMe5aEgrY5E1C2UYIjo4Ft2fxsJ
QgxeeIF4Kww/Bhh/gHd/FQcNIZCJeB8OAyaYChKhEiJfJAMDJJAioqGkLSMEDCCQtC2bBha1kBMG
PwwrumoqjD8IHMFgGQtyKMgnAUQAD821Jw7Q0QEbwHgqGgHYTQgMFCAkJB8UDMtZRAUNpw29TUEA
Ow=="
	    }
	    150 {
		set data "
R0lGODlhGAAYAKUyABEREYoAAIsAAIwAAIsBAYwBAYsCAowCAowDA40DA40FBY4FBY4GBpwlJUBA
QJ0mJp0nJ54oKJ4pKURERElJSaM0NKM1NaM2NqQ3N6Q4OLFTU7FUVLJVVbNXV7NYWLNZWbRbW75x
cb9ycr9zc8B0dMB1dZKSku7Z2e7a2u/b2+/c3PDe3vDf3/Hg4PHh4erq6vv39/z4+P//////////
/////////////////////////////////////////////yH5BAEKAD8ALAAAAAAYABgAAAbbwJ9w
OCwoDgcFgchsDgiCCCgUAkEE0CbTwCCpZOCwasQ4aIWFjCsse73YrUtBW+DE2DIKBQ/bzIkGGHd4
Dg54MjAWZkIDDGuHhYcyLAoCQgQlkjKRkiJLPwIompyHKZYEEpqbhpoPRiCqpIceCgYksQATurom
YSMGCSOqJru7vWAiCAofqs0yHQsEEc6qD0sCJ9Sllj8Et9psIX8CDC3gYCsKAUMHFzDgMBWLRRvv
zjAaf0wFFizNKxX0NUkiQhQeFCGOnBmCRcCDD1Q+NBBQYN3CfQsMGFggkEkQADs="
	    }
	    175 {
		set data "
R0lGODlhHAAcAKUlAAQEBIoAAIsAAIwAAIsBAYwBASAgIIwCAowDAyIiIo4FBY4GBpUWFpYYGJcZ
GZcaGqIyMqIzM6M0NGpqarNXV7NYWLNZWbVdXbZeXqGhodOentOfn9msrNqtrdqurtuvr+K/v/Dd
3fDe3vr09Pr19f//////////////////////////////////////////////////////////////
/////////////////////////////////////////////yH5BAEKAD8ALAAAAAAcABwAAAb3wJ9w
SAwMBASC4EhsOocFgQDhiEQcCGnh+ZQqMB5SaVwieTAKQYA7VFpE5Dg5VEGyC4uNvJTJ7DcLW04E
CyB7JQkJhyALB01SeocGBoclG2pEAhaVJZOcdVAKIZyelSEKBEICGJydlJwXAj8BAh+tpZUeUgMI
YqSvlSNZAg+trsYNSBLGBgCJzwkTcREEBRHGE9DP0mQQSg3G4SUMSwgj4pzCRwIe6JUcsj+r7oex
QgWi9HGnqaoV+mQoxBNCKxI9DVKaEDLkjpEjJ3gMGtMQiI2SCqM40bHDZpaALx3OkRnRAU3CjlCk
HGhgpcEBLSi7HEkiZeCTIAA7"
	    }
	    200 {
		set data "
R0lGODlhIAAgAKUyAAAAAIoAAIsAAIwAAIsBAZogIDw8PJoiIpskJKAvL6EwMKExMaIyMqIzM6M0
NKY8PKg/P6hAQKlBQalCQrBSUrFUVLJVVbJWVsV/f8WAgMaBgcaCguLAwOPBweTDw+XFxebHx+bI
yOfKyuzW1u3X1+3Y2O7Z2fTm5vTn5/Xo6PXp6fXq6vbr6/bs7Pft7fz4+Pz5+f36+v//////////
/////////////////////////////////////////////yH5BAEKAD8ALAAAAAAgACAAAAb+wJ9w
SBQGBMikoMhsDo/KKDLgdCYJjY0nFYulPBqGskocCAgXk2zNbpcsBAGVjESI2nh86ICsIh8reTIG
BoIuE31MZhIxgjIAAI4wEXJFdS2Oj5GOLHxzQmd3mZCZMiBxQ0gXpZqsFYlnaqWkpSSoPwIOrK2s
C0tHG7u0pRlKHsKbpR1KKci7J0owyJDUkIVsL9G7BtXV12vZSc275NACZsfku8tQGuq7GOe4DO+s
CUu4BCX1jiOwAhb4CaKQKF8IgW0+3AIl4AALhDJYFMBH5EiERvxgQChI5I+Kei0QUWSC5AAIdR8m
jmxyhECFfZlIUIiz0gkUAQsydEDx4gUTCg4YFCQh00SK0QFEyZiJgpRoEAA7"
	    }
	}
	scaleutilmisc_watchImg put $data -format gif
    }

    return $watchImg
}

#------------------------------------------------------------------------------
# scaleutilmisc::scaleIncrComboboxArrows
#------------------------------------------------------------------------------
proc scaleutilmisc::scaleIncrComboboxArrows pct {
    switch $pct {
	100 {
	    downarrow configure -data "
#define down_width 16
#define down_height 16
static unsigned char down_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xfc, 0x7f, 0xf8, 0x3f, 0xf0, 0x1f, 0xe0, 0x0f, 0xc0, 0x07, 0x80, 0x03,
   0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"

	    uparrow configure -data "
#define up_width 16
#define up_height 16
static unsigned char up_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0xc0, 0x01, 0xe0, 0x03,
   0xf0, 0x07, 0xf8, 0x0f, 0xfc, 0x1f, 0xfe, 0x3f, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	}

	125 {
	    downarrow configure -data "
#define down_width 20
#define down_height 20
static unsigned char down_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0xff, 0x07,
   0xf8, 0xff, 0x03, 0xf0, 0xff, 0x01, 0xe0, 0xff, 0x00, 0xc0, 0x7f, 0x00,
   0x80, 0x3f, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x04, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"

	    uparrow configure -data "
#define up_width 20
#define up_height 20
static unsigned char up_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x02, 0x00, 0x00, 0x07, 0x00, 0x80, 0x0f, 0x00, 0xc0, 0x1f, 0x00,
   0xe0, 0x3f, 0x00, 0xf0, 0x7f, 0x00, 0xf8, 0xff, 0x00, 0xfc, 0xff, 0x01,
   0xfe, 0xff, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	}

	150 {
	    downarrow configure -data "
#define down_width 24
#define down_height 24
static unsigned char down_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0xf8, 0xff, 0x3f, 0xf0, 0xff, 0x1f, 0xe0, 0xff, 0x0f,
   0xc0, 0xff, 0x07, 0x80, 0xff, 0x03, 0x00, 0xff, 0x01, 0x00, 0xfe, 0x00,
   0x00, 0x7c, 0x00, 0x00, 0x38, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"

	    uparrow configure -data "
#define up_width 24
#define up_height 24
static unsigned char up_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x3e, 0x00,
   0x00, 0x7f, 0x00, 0x80, 0xff, 0x00, 0xc0, 0xff, 0x01, 0xe0, 0xff, 0x03,
   0xf0, 0xff, 0x07, 0xf8, 0xff, 0x0f, 0xfc, 0xff, 0x1f, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	}

	175 {
	    downarrow configure -data "
#define down_width 28
#define down_height 28
static unsigned char down_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0xff, 0xff, 0x03,
   0xf0, 0xff, 0xff, 0x01, 0xe0, 0xff, 0xff, 0x00, 0xc0, 0xff, 0x7f, 0x00,
   0x80, 0xff, 0x3f, 0x00, 0x00, 0xff, 0x1f, 0x00, 0x00, 0xfe, 0x0f, 0x00,
   0x00, 0xfc, 0x07, 0x00, 0x00, 0xf8, 0x03, 0x00, 0x00, 0xf0, 0x01, 0x00,
   0x00, 0xe0, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00};"

	    uparrow configure -data "
#define up_width 28
#define up_height 28
static unsigned char up_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
   0x00, 0x70, 0x00, 0x00, 0x00, 0xf8, 0x00, 0x00, 0x00, 0xfc, 0x01, 0x00,
   0x00, 0xfe, 0x03, 0x00, 0x00, 0xff, 0x07, 0x00, 0x80, 0xff, 0x0f, 0x00,
   0xc0, 0xff, 0x1f, 0x00, 0xe0, 0xff, 0x3f, 0x00, 0xf0, 0xff, 0x7f, 0x00,
   0xf8, 0xff, 0xff, 0x00, 0xfc, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00};"
	}

	200 {
	    downarrow configure -data "
#define down_width 32
#define down_height 32
static unsigned char down_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xf8, 0xff, 0xff, 0x3f, 0xf0, 0xff, 0xff, 0x1f, 0xe0, 0xff, 0xff, 0x0f,
   0xc0, 0xff, 0xff, 0x07, 0x80, 0xff, 0xff, 0x03, 0x00, 0xff, 0xff, 0x01,
   0x00, 0xfe, 0xff, 0x00, 0x00, 0xfc, 0x7f, 0x00, 0x00, 0xf8, 0x3f, 0x00,
   0x00, 0xf0, 0x1f, 0x00, 0x00, 0xe0, 0x0f, 0x00, 0x00, 0xc0, 0x07, 0x00,
   0x00, 0x80, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"

	    uparrow configure -data "
#define up_width 32
#define up_height 32
static unsigned char up_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x80, 0x00, 0x00, 0x00, 0xc0, 0x01, 0x00, 0x00, 0xe0, 0x03, 0x00,
   0x00, 0xf0, 0x07, 0x00, 0x00, 0xf8, 0x0f, 0x00, 0x00, 0xfc, 0x1f, 0x00,
   0x00, 0xfe, 0x3f, 0x00, 0x00, 0xff, 0x7f, 0x00, 0x80, 0xff, 0xff, 0x00,
   0xc0, 0xff, 0xff, 0x01, 0xe0, 0xff, 0xff, 0x03, 0xf0, 0xff, 0xff, 0x07,
   0xf8, 0xff, 0xff, 0x0f, 0xfc, 0xff, 0xff, 0x1f, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};"
	}
    }
}
